import java.io.*;
import java.net.*;
import java.nio.CharBuffer;
import java.util.*;

public class Server {

    private final Controller controller;
    public static FileSystem fileSystem;
    public static int pos;

    public Server(Controller controller, FileSystem fileSystem) {
        this.controller = controller;
        Server.fileSystem = fileSystem;
    }

    public void handleClient(Socket client) throws IOException {
        OutputStream out = client.getOutputStream();
        InputStream in = client.getInputStream();
        pos = 0;

        BufferedReader req = new BufferedReader(new InputStreamReader(in));
        PrintWriter res = new PrintWriter(new OutputStreamWriter(out));

        System.out.println("---------NEW CONNECTION---------");

        String line;
        while((line = req.readLine()) != null) {
            String[] tokens = line.split(" ");
            String command = tokens[0];

            if (command.equals("JOIN")) {
                int publicPort = Integer.parseInt(tokens[1]);
                System.out.println("Dstore wants to connect");
                FSStore fsStore = new FSStore(client, publicPort);
                fileSystem.addDstore(client.getPort(), fsStore);
            } else if (fileSystem.getDstores().size() < controller.getR()) {
                res.println("ERROR_NOT_ENOUGH_DSTORES");
                res.flush();
            } else {
                if (fileSystem.getDstores().containsKey(client.getPort())) {
                    if(command.equals("STORE_ACK")) {
                        handleStoreACK(tokens);
                    } else if (command.equals("REMOVE_ACK")){
                        handleRemoveACK(tokens);
                    } else {
                        System.out.println("Unknown command");
                    }
                } else {
                    switch (command) {
                        case "STORE" -> handleStore(client, tokens);
                        case "LOAD" -> handleLoad(client, tokens);
                        case "RELOAD" -> handleReload(client, tokens);
                        case "REMOVE" -> handleRemove(client, tokens);
                        case "LIST" -> handleList(client);
                        default -> System.out.println("Unknown command");
                    }
                }
            }
        }
    }

    /**
     * @param client
     * @param tokens gets the filename and the filesize
     * @throws IOException
     */
    private void handleStore(Socket client, String[] tokens) throws IOException {
        try {

            String filename = tokens[1];
            int filesize = Integer.parseInt(tokens[2]);
            PrintWriter res = new PrintWriter(new OutputStreamWriter(client.getOutputStream()));

            if(fileSystem.getStore().containsKey(filename)) {
                res.println("ERROR_FILE_ALREADY_EXISTS");
                res.flush();
                return;
            }

            // update index to store in progress
            fileSystem.addIndex(filename, "store in progress");

            // select R Dstores and create a string of all of their endpoints
            String msg = "";
            int i = 0;
            List<FSStore> temp = new ArrayList<>();

            for(int port : fileSystem.getDstores().keySet()) {
                if(i == controller.getR()) {
                    break;
                }

                // in case all the ACK are received keep in memory all the dstores
                temp.add(fileSystem.getDstores().get(port));

                // construct the message
                msg = fileSystem.getDstores().get(port).getPublicPort() + " " + msg;
                i++;
            }

            res.println("STORE_TO " + msg);
            res.flush();

            // check if all the dstores have sent an ACK and send appropriate messages
            boolean done = false;
            long limit = System.currentTimeMillis() + controller.getTimeout();

            while(!done && System.currentTimeMillis() < limit) {
                if(fileSystem.getStore().containsKey(filename) && fileSystem.getStore().get(filename).size() == controller.getR()) {
                    done = true;
                    fileSystem.addIndex(filename, "store complete");
                    fileSystem.addStore(filename, temp);
                    fileSystem.addFileSize(filename, filesize);
                    res.println("STORE_COMPLETE");
                    res.flush();
                }
            }

            // if the dstores didn't send a STORE_ACK in the timeout => store failed
            if(!done) {
                fileSystem.removeIndex(filename);
                System.out.println(filename + " failed to upload");
            }

        } catch (IndexOutOfBoundsException e) {
            System.out.println("Arguments don't match in STORE operation");
        }
    }

    private void handleLoad(Socket client, String[] tokens) throws IOException {

        try {
            String filename = tokens[1];
            PrintWriter res = new PrintWriter(new OutputStreamWriter(client.getOutputStream()));
            if(!fileSystem.getStore().containsKey(filename)) {
                res.println("ERROR_FILE_DOES_NOT_EXIST");
            } else {
                // select a Dstore from there and give an appropriate error if all Dstores fail
                res.println("LOAD_FROM " + fileSystem.getStore().get(filename).get(pos).getPublicPort() +
                        " " + fileSystem.getFileSizes().get(filename));
            }

            res.flush();
        } catch (IndexOutOfBoundsException e) {
            System.out.println("Arguments don't match in LOAD operation");
        }
    }

    private void handleReload(Socket client, String[] tokens) throws IOException {
        pos = pos + 1;
        PrintWriter res = new PrintWriter(new OutputStreamWriter(client.getOutputStream()));

        try {
            String filename = tokens[1];
            if(!fileSystem.getStore().containsKey(filename)) {
                pos = 0;
                res.println("ERROR_FILE_DOES_NOT_EXIST");
            } else {

                if(pos == controller.getR()) {
                    pos = 0;
                    res.println("ERROR_LOAD");
                    res.flush();
                    return;
                }

                // select a Dstore from there and give an appropriate error if all Dstores fail
                res.println("LOAD_FROM " + fileSystem.getStore().get(filename).get(pos).getPublicPort() +
                        " " + fileSystem.getFileSizes().get(filename));
            }

            res.flush();
        } catch (IndexOutOfBoundsException e) {
            System.out.println("Arguments don't match in LOAD operation");
        }
    }

    private void handleRemove(Socket client, String[] tokens) throws IOException {
        try {

            String filename = tokens[1];

            fileSystem.addIndex(filename, "remove in progress");
            System.out.println(fileSystem.index.get(filename));

            if(!fileSystem.store.containsKey(filename)) {
                PrintWriter res = new PrintWriter(new OutputStreamWriter(client.getOutputStream()));
                res.println("ERROR_FILE_DOES_NOT_EXIST");
                res.flush();
                return;
            }

            for(FSStore fsStore : fileSystem.getStore().get(filename)) {
                System.out.println("Sending to Dstore");
                PrintWriter res = fsStore.getOutput();
                res.println("REMOVE " + filename);
                res.flush();
            }

            boolean done = false;
            long limit = System.currentTimeMillis() + controller.getTimeout();
            PrintWriter res = new PrintWriter(new OutputStreamWriter(client.getOutputStream()));

            while(!done && System.currentTimeMillis() < limit) {
                if(fileSystem.getStore().get(filename).isEmpty()) {
                    done = true;
                    fileSystem.addIndex(filename, "remove complete");
                    fileSystem.getStore().remove(filename);
                    fileSystem.getFileSizes().remove(filename);
                    res.println("REMOVE_COMPLETE");
                    res.flush();
                }
            }

            if(!done) {
                System.out.println(filename + " failed to remove");
            }

        } catch (IndexOutOfBoundsException e) {
            System.out.println("Arguments don't match in REMOVE operation");
        }
    }

    private void handleList(Socket client) throws IOException {
        String msg = "";
        for(String filename : fileSystem.getStore().keySet()) {
            msg = filename + " " + msg;
        }

        PrintWriter res = new PrintWriter(new OutputStreamWriter(client.getOutputStream()));
        res.println("LIST " + msg);
        res.flush();
    }

    private void handleStoreACK(String[] tokens) {

        String filename = tokens[1];
        int dstorePort = Integer.parseInt(tokens[2]);

        if(fileSystem.getStore().containsKey(filename)) {
            fileSystem.getStore().get(filename).add(fileSystem.getDstores().get(dstorePort));
        } else {
            List<FSStore> d = new ArrayList<>();
            d.add(fileSystem.getDstores().get(dstorePort));
            Server.fileSystem.addStore(filename,d);
        }

    }

    private void handleRemoveACK(String[] tokens) {
        String filename = tokens[1];
        for(FSStore fsStore : fileSystem.getStore().get(filename)) {
            if(!fsStore.getFiles().contains(filename)) {
                fileSystem.removeDstore(fsStore, filename);
                break;
            }
        }
    }

}
