#!/usr/bin/env python

# vgen.py - Various functions for working with lists of things (pins, regs, whatever)
# Paul Whatmough Jan 2014
# Paul Whatmough Nov 2016 - Reworked to move to CSV

import time;
import re;
import shutil;
import os;
import csv;

# File format for signal list is comma delimited fields, 
# each field takes on the key given in a header on the first
# line of the file.
#
# See each script for the required field/key values.
#
# The CSV file is read in and stored in a vglist structure.
# Vglist (vgl) is just a list of dictionaries.
# Each element of the list is an object, such as a register or a pin.


###############################################################################
# Working with Verilog
###############################################################################


def reg_dims(reg):
  """ 
  Accepts a dict containing the fields 'start' and 'nbits'.
  Returns a string containing the dims in verilog syntax.
  If nbits is 0, returns empty string.
  """
  if (int(reg['nbits']) == 0):
    return ''
  else:
    return "["+str(int(reg['nbits'])+int(reg['start'])-1)+":"+reg['start']+"]"

def banner_start(): 
  return "// VGEN START: Autogenerated by "+__file__+" on "+time.strftime("%H:%M:%S %d/%m/%Y")+"\n\n"

def banner_end(): 
  return "// VGEN END: Autogenerated by "+__file__+" on "+time.strftime("%H:%M:%S %d/%m/%Y")+"\n\n"

def read_to_tag(fi,tag):
  foundit = tag == ''
  outstr = ""
  for line in fi:
    outstr += line
    if (tag in line) and (tag !=""):
      foundit = True
      break
  assert foundit, 'Reached end of file without finding tag!  Tag: %s' % tag
  return outstr


def get_verilog_signals(module_file,signal_prefix):
  """ 
  Read a (system)verilog file and return all signals with a matching prefix, along with some related details 
  """
  fi = open(module_file,"r")
  print "** Reading file \""+module_file+"\" to get signals with prefix \""+signal_prefix+"\"."
  
  # define some regexs
  re_nb = '(\\b'+re.escape(signal_prefix)+'\w+)\[(\d+)\:(\d+)\]'    # this regex matches multibit signals: e.g. "mysig[7:0]"
  #re_nb = '(\\bd_\w+)\[(\d+)\:(\d+)\]'    # this regex matches multibit signals: e.g. "mysig[7:0]"
  re_1b = '(\\b'+re.escape(signal_prefix)+'\w+\\b)'                 # this regex matches single bit signals: e.g "mysig" 
  #re_1b = '(\\bd_\w+\\b)'                 # this regex matches single bit signals: e.g "mysig" 
  
  # check file line by line, checking for matches
  found = []
  keys = ['name','nbits']
  for line in fi:
    if line.lstrip().startswith('//') or (line == '') or line.isspace():
      pass
    else:
      if re.search(re_nb,line):                     # found a signal with multiple bits
        signame = re.search(re_nb,line).group(1)    # get all the signal attributes
        nhi = int(re.search(re_nb,line).group(2))
        nlo  = int(re.search(re_nb,line).group(3))
        nbits  = ((int(nhi)+1) - int(nlo))
        assert nlo == 0                             # check the vector range starts at 0
        new_dict = dict(zip(keys,[signame,nbits]))
        found.append(new_dict)
      elif re.search(re_1b,line):                   # found a signal with a single bit
        signame = re.search(re_1b,line).group(1)
        nbits = 1
        new_dict = dict(zip(keys,[signame,nbits]))
        found.append(new_dict)
 
  fi.close()

  # Remove duplicates from found signals
  found = remove_duplicates(found,'name')
  
  return found


def read_to_verilog_module_declaration(fi):
  """ Read up to and including module declaration in a (system)verilog file. """
  
  # Looking for a string of this format: "module NAME ("
  # The three componenets may be split over multiple lines.
  # Assuming that there will be no interesting text after the open bracket.
  
  # First look for the word 'module'
  for line in fi:
    line = line.lstrip()
    if line and not line.startswith('//'): 
      if 'module' in line:
        #print 'found module in this line: %s' % line
        if '(' in line:
          #print 'found ( in this line: %s' % line
          return
        else:
          break
  # If the open bracket was not on the same line as the word 'module',
  # keep looking for the open bracket.
  for line in fi:
    line = line.lstrip()
    if line and not line.startswith('//'): 
      if '(' in line:
        #print 'found ( in this line: %s' % line
        break
  return




def get_verilog_module_signals(module_file,debug=False):
  """
  Read in a (system)verilog module and get all the signals in the module declaration.
  Returns a list of dictionaries, (vglist).  
  """

  d = debug

  # This is all the data that can be extracted from the module declaration.
  keys = [
    'name',       # logical name for pad
    'direction',  # direction: input or output
    'nbits',      # number of bits
    'start'       # start bit (generally assumed to be zero)
  ]

  fi = open(module_file,"r")
  
  # Read up to the module declaration open bracket
  read_to_verilog_module_declaration(fi)
  
  # N.B. this is not actually parsing the code, so beware of changes in style!
  # Expecting the following style: " input/output logic [X:Y] NAME ,"
  vglist = []
  re_1b = '(input|output)\s+logic\s+(\w+)'
  re_nb = '(input|output)\s+logic\s+\[(\d+):(\d+)\]\s+(\w+)'
  for line in fi:
    if line.lstrip().startswith('//') or (line == '') or line.isspace():
      pass
    else:
      if ')' in line:
        break
      elif re.search(re_nb,line): 
        if d: print 'Found nb signal in module declaration.'
        nbits = int(re.search(re_nb,line).group(2)) - int(re.search(re_nb,line).group(3)) +1
        start = int(re.search(re_nb,line).group(3)) 
        assert start == 0, 'start bit index in slice is not zero: %s' % start
        signal_match = [re.search(re_nb,line).group(4).strip(),re.search(re_nb,line).group(1).strip(),nbits,0]
      elif re.search(re_1b,line): 
        if d: print 'Found 1b signal in module declaration.'
        nbits = 1
        signal_match = [re.search(re_1b,line).group(2).strip(),re.search(re_1b,line).group(1).strip(),nbits,0]
      new = dict((key,val) for key,val in zip(keys,signal_match))
      if d: print new
      vglist.append(new)
 
  return vglist

 


###############################################################################
# Working with CSV
###############################################################################


def read_csv(csv_file, debug=False):
  """
  Reads in a CSV file.  
  Returns a list of dictionaries, (vglist).  
  Keys are derived from the first line of the csv file which contains a header.
  """
  d = debug

  # Read in signal list (if it exists)
  try:
    fi = open(csv_file,"r")
  except IOError:
    print "** Error: File not found: "+vglist_file
    sys.exit()
  
  # create an empty list to add dictionaries to
  vglist = [] 
 
  # use csv module to iterate over input file
  reader = csv.reader(fi)
  
  # get keys from header on first line
  row = next(reader) 
  header = [x.strip(' ') for x in row]      # clean up any whitespace
  header = [header[0].replace('*','').strip()] + header[1:]
  if d: print "line contains key list: " + str(header)
 
  # get the rows
  for row in reader:
    if d: print row
    if not row:
      if d: print "line is empty"
    elif not row[0].strip(): 
      if d: print "line is whitespace"
    #elif row[0].strip().startswith('*'):            # header starts with * (store this to generate keys)
    #  header = [x.strip(' ') for x in row]          # clean up any whitespace
    #  header = [header[0].replace('*','').strip()] + header[1:]
    #  if d: print "line contains key list: " + str(header)
    elif row[0].lstrip().startswith("#"):            # comments or whitespace (ignore) 
      pass
      if d: print "line is comment: "+"-".join(row)
    else:                                           # Add legit entry to list
      assert len(header) > 0                        # should have already seen the header 
      entry = [x.strip(' ') for x in row[:]]        # clean up any whitespace
      assert len(header) == len(entry)              # check equal number of key/val pairs
      if d: print "line is CSV row: "+str(entry)
      new = dict((key,val) for key,val in zip(header,entry))
      if d: print "new dict is: "+str(new)
      vglist.append(new)
  
  #print vglist
  fi.close()
  
  return vglist



def append_csv(csv_file,vglist,keys,unused_str='?',debug=False):
  """
  Append the list of dicts to csv_file.
  Each element of list is written as a CSV row.
  The dict elements and ordering is according to keys argument.
  unused_str is used to fill fields in CSV that are not in vglist.
  """
  d = debug
  assert vglist != [], 'empty vglist supplied: %s' %str(vglist)

  print "** Appending new signals to csv file \""+csv_file+"\""
  if os.path.isfile(csv_file):
    shutil.copy2(csv_file,csv_file+".bak")   # copy2 preserves mod/access info
 
  # First read the header line to get the order of the keys
  fi = open(csv_file,'r')
  reader = csv.reader(fi)
  row = next(reader) 
  header = [x.strip(' ') for x in row]      # clean up any whitespace
  header = [header[0].strip()] + header[1:]
  assert len(header) > 0, 'Header is missing in CSV file: %s' % csv_file
  assert all(x in header for x in keys), 'Header does not contain all keys.  Header: %s, Keys: %s' % (header,keys)
  if d: print "Header line contains key list: " + str(header)
  fi.close()
   
  # Now append new rows to CSV, obeying order specified in header.
  fo = open(csv_file,'ab')
  fo.write("\n# New signals:\n")
  writer = csv.writer(fo, lineterminator='\n')
  for row in vglist:
    line = []
    for item in header:
      if item in row.keys():
        line.append(row[item])
      else:
        line.append(unused_str)
    #line = [row[x] for x in keys]
    if d: print line
    writer.writerow(line) 
  fo.close()




def find_new(vglist1,vglist2,key):
  """
  Return list of elements of vglist2 that are not in vglist1.
  Uses specified key to determine new elements of list.
  """
  # Get all the values for the specified key
  values = []
  for row in vglist1:
    values.append(row[key])
  
  # Check through the new list for new values
  new_rows = []
  for row in vglist2:
    if row[key] in values:
      # already exists in vglist1
      continue  
    else:
      new_rows.append(row)

  return new_rows



def remove_key(vglist,key):
  """
  Accepts a list of dictionaries (vglist) and a list of keys.
  Returns a list of dictionaries with each of the specified keys removed for all element of original list.
  """
  new_list = []
  for row in vglist:
    for item in key:
      row.pop(item,None)
    new_list.append(row)
  return new_list 


def remove_duplicates(vglist,key):
  """
  Find duplicates for the specified key and keep only the first one.
  The returned list should contain only unique entries for specified key.
  """
  # TODO would be nice to also optionally clear the other fields for the duplicates
  new_list = []
  values = []
  for row in vglist:
    val = row[key]
    if val in values:
      # duplicate!
      print "**Found duplicate: "+row
    else:
      new_list.append(row)
      values.append(row[key])
  return new_list 



def check_field(vglist,key,value):
  """
  Accepts a list of dicts; for each dict, check the the specified key contains the specified value.
  """
  for row in vglist:
    assert row[key] == value, \
      '** Error: Field: %s contains value: %s, expected: %s' % (str(key),str(row[key]),str(value))
  return True

def check_complete(vglist):
  """
  Accepts a list of dicts and checks no values are empty ("").
  """
  for row in vglist:
    l = row.values()
    for field in l:
      if (field == ""):
        print "** Warning: Signals list contains incomplete fields!"
        return False 
  return True



def check_keys_exist(vglist,my_keys):
  """
  Accepts a list of dicts and checks the specified keys exist in each list element.
  """
  for d in vglist:
    if not all(key in d for key in my_keys): 
      return False
  return True




###############################################################################
#
###############################################################################




def test():
  """ 
  Simple test of some of the functions
  """

  # TODO test of the verilog module reading function
  print get_verilog_module_signals('../TOP.sv',debug=True)
  
  # Test reading in a list from a CSV file
  myvglist = read_csv('test.csv')
  assert check_complete(myvglist)
  assert check_keys(myvglist,['this','that'])

  #print test_list
  #for x in test_list:
  #  print x["this"]
  #for x in test_list:
  #  print x["that"]

  # Test check keys
  print check_keys(myvglist,['this'])
  print check_keys(myvglist,['this','that'])
  print check_keys(myvglist,['this','that','other'])
  
  # Test appending to CSV file
  append_csv('test.csv',[{'this':'blah','that':'blaf'}],['this','that'])

if __name__ == "__main__":
    test()



