#include <qbluetoothaddress.h>
#include <qbluetoothdevicediscoveryagent.h>
#include <qbluetoothlocaldevice.h>
#include <qbluetoothdeviceinfo.h>
#include <qbluetoothservicediscoveryagent.h>
#include <QDebug>
#include <QList>
#include <QMetaEnum>
#include <QTimer>
#include <QSignalMapper>
#include <QDebug>
#include <iostream>
#include "bleController.h"
#include "serviceInfo.h"
#include "deviceInfo.h"
#include "charInfo.h"

BleController::BleController(){
    discoveryAgent = new QBluetoothDeviceDiscoveryAgent();
    discoveryAgent->setLowEnergyDiscoveryTimeout(5000);
    connect(discoveryAgent, &QBluetoothDeviceDiscoveryAgent::deviceDiscovered,
             this, &BleController::addNewDevice);
    connect(discoveryAgent, &QBluetoothDeviceDiscoveryAgent::finished, this, &BleController::scanDone);
    connect(discoveryAgent, QOverload<QBluetoothDeviceDiscoveryAgent::Error>::of(&QBluetoothDeviceDiscoveryAgent::error),
             this, &BleController::scanError);

    for(int i = 0; i<5; i++){
        finalScannedServices[i] = nullptr;
    }
    //Add the QLists according to number of devices

    // for(int i = 0; i<5; i++){
    //     scannedServices.append(QList<ServiceInfo*>());
    //     scannedCharacteristics.append(QList<CharacteristicInfo*>());
    //     qDebug() << scannedServices.length() <<Qt::endl;
    //     qDebug() << scannedCharacteristics.length() <<Qt::endl;
    //     // qDebug() << scannedServices[i].length() <<Qt::endl;
    //     // qDebug() << scannedCharacteristics[i].length() <<Qt::endl;
    //     // scannedCharacteristics[0].clear();
    //     // scannedServices[0].clear();
    // }


}

BleController::~BleController(){
    delete discoveryAgent;
    qDeleteAll(scannedDevices);
    qDeleteAll(initialScannedServices);



    for(QList<ServiceInfo*> tempServiceList:scannedServices){
        tempServiceList.clear();
    }
    for(QList<CharacteristicInfo*> tempCharList:scannedCharacteristics){
        tempCharList.clear();
    }
    scannedDevices.clear();
    initialScannedServices.clear();
    // scannedCharacteristics.clear();
}

int BleController::startDeviceScan(){

    qDebug() << "Scanning devices..." << Qt::endl;
    //Clear devices list
    qDeleteAll(scannedDevices);
    scannedDevices.clear();
    emit deviceListUpdated();
    //start the scan
    discoveryAgent->start(QBluetoothDeviceDiscoveryAgent::LowEnergyMethod);
    if (discoveryAgent->isActive()) {
        m_deviceScanState = true;
        emit stateChanged(true);
    }
    return 30;
}

void BleController::addNewDevice(const QBluetoothDeviceInfo &info)
{
    if (info.coreConfigurations() & QBluetoothDeviceInfo::LowEnergyCoreConfiguration){
        int index = scannedDevices.length();
        qDebug() << "Last device added: " << info.name() << " device index: " << index << Qt::endl;
        if(info.coreConfigurations() & QBluetoothDeviceInfo::LowEnergyCoreConfiguration){
            scannedDevices.append(new DeviceInfo(info));
        }

        emit newDeviceAdded(index, info.name());
    }
}

void BleController::scanDone(){
    qDebug() << "Device scan finished" << Qt::endl;
    const QList<QBluetoothDeviceInfo> discoveredDevices = discoveryAgent->discoveredDevices();
    for (auto newDevice : discoveredDevices)
        if (newDevice.coreConfigurations() & QBluetoothDeviceInfo::LowEnergyCoreConfiguration)
            scannedDevices.append(new DeviceInfo(newDevice));

    emit deviceListUpdated();
    m_deviceScanState = false;
    emit stateChanged(false);
    
    if (scannedDevices.isEmpty())
        qDebug() << "No devices found during scan " << Qt::endl;
}

void BleController::scanError(QBluetoothDeviceDiscoveryAgent::Error error){

    qDebug() << "Device scan error" << Qt::endl;
    //Print out the error we find
    QString errorStr;
    if (error == QBluetoothDeviceDiscoveryAgent::PoweredOffError){
        qDebug() << "BT adapter powered off" << Qt::endl;
        errorStr ="The Bluetooth adaptor is powered off, power it on before doing discovery.";

    }
    else if (error == QBluetoothDeviceDiscoveryAgent::InputOutputError)
        errorStr = "Writing or reading from the device resulted in an error.";
    else {
        static QMetaEnum qme = discoveryAgent->metaObject()->enumerator(
                    discoveryAgent->metaObject()->indexOfEnumerator("Error"));
        errorStr = "Error: " + QLatin1String(qme.valueToKey(error));
    }

    //Emit update signals
    m_deviceScanState = false;
    emit deviceListUpdated();
    emit stateChanged(false);
    emit scanErrorSignal(errorStr);
}


void BleController::startInitialServiceScan(int index){

    qDebug() << "Starting initial service scan" <<Qt::endl;
    // Find a device from the list according to the input index
    initialScanTempDevice.setDevice((scannedDevices[index])->getDevice());
    qDebug() << "Doing service scan on device name: " <<initialScanTempDevice.getName() << Qt::endl;

    if (!initialScanTempDevice.getDevice().isValid()) {
        qWarning() << "Not a valid device";
        return;
    }

    //Clean up service and characterstic lists before assigning to them
    // qDeleteAll(scannedCharacteristics);
    // scannedCharacteristics.clear();
    qDeleteAll(initialScannedServices);
    initialScannedServices.clear();


    // if (initialScanController && m_previousAddress != initialScanTempDevice.getAddress()) {
    //     qDebug() << "Device address mismatch during service scan" << Qt::endl;
    //     initialScanController->disconnectFromDevice();
    //     delete initialScanController;
    //     initialScanController = nullptr;
    // }

    if(initialScanController!=nullptr){
        initialScanController->disconnectFromDevice();
        delete initialScanController;
        initialScanController = nullptr;
    }

    // if (!initialScanController) {

        qDebug() << "Creating service scan initialScanController" <<initialScanTempDevice.getName() << Qt::endl;
        // Connecting signals and slots for connecting to LE services.
        initialScanController = QLowEnergyController::createCentral(initialScanTempDevice.getDevice());
        connect(initialScanController, &QLowEnergyController::connected,
                this, &BleController::initialServiceScanDeviceConnected);
        connect(initialScanController, QOverload<QLowEnergyController::Error>::of(&QLowEnergyController::error),
                this, &BleController::initialServiceScanErrorRecieved);
        connect(initialScanController, &QLowEnergyController::disconnected,
                this, &BleController::initialServiceScanDeviceDisconnected);
        connect(initialScanController, &QLowEnergyController::serviceDiscovered,
                this, &BleController::initialServiceScanAddLowEnergyService);
        connect(initialScanController, &QLowEnergyController::discoveryFinished,
                this, &BleController::initialServiceScanDone);
    // }

    if (isRandomAddress())
        initialScanController->setRemoteAddressType(QLowEnergyController::RandomAddress);

    else
        initialScanController->setRemoteAddressType(QLowEnergyController::PublicAddress);
    initialScanController->connectToDevice();

    m_previousAddress = initialScanTempDevice.getAddress();

}

void BleController::initialServiceScanDeviceConnected()
{
    qDebug() << "Device connected" << Qt::endl;
    initialScanController->discoverServices();
}

void BleController::initialServiceScanErrorRecieved(QLowEnergyController::Error)
{
    qWarning() << "Error: " << initialScanController->errorString();
}

void BleController::initialServiceScanDeviceDisconnected()
{
    qWarning() << "Disconnect from device";
    emit disconnectedFromInitialScanDevice();
}

void BleController::initialServiceScanAddLowEnergyService(const QBluetoothUuid &serviceUuid)
{
    qDebug() << "New service found" << Qt::endl;
    QLowEnergyService *service = initialScanController->createServiceObject(serviceUuid);
    if (!service) {
        qWarning() << "Cannot create service for uuid";
        return;
    }
    auto serv = new ServiceInfo(service);
    initialScannedServices.append(serv);

    emit servicesUpdated();
    ServiceInfo* tempInfo = qobject_cast<ServiceInfo *>(serv);
    qDebug() << "ServiceName: " << tempInfo->getName() << Qt::endl;
}

void BleController::initialServiceScanDone()
{
    qDebug() << "Initial service scan done... Seeing if device is compatible";
    // QString serviceUuid = initialScannedServices[1]->getUuid();
    // qDebug() << "Main service UUID: " << servInfo << Qt::endl;

    //Determine device type by the main service uuid
    for(ServiceInfo *servInfo:initialScannedServices){
        qDebug() << "Service scanned: " << servInfo->getUuid() << Qt::endl;
        if(QString::compare(servInfo->getUuid(), wristwatchServiceUUID) == 0){
            qDebug() << "Pairing with wristwatch device";
            //Check if wristwatch isnt already connected
            for(int i = 0; i <pairedDevices.count(); ++i){
                qDebug() << "Index: "<< i <<Qt::endl;
                if(pairedDevices[i]->devType == DeviceType::Wristwatch){
                    qDebug() << "A wristwatch is already paired. Unpair it first before pairing this one" << Qt::endl;

                    return;
                }
            }

            //No wristwatch device already paired, proceed to pair this one
            DeviceInfo *tempDevInfo = new DeviceInfo(initialScanTempDevice.getDevice());
            tempDevInfo->devType = DeviceType::Wristwatch;
            pairedDevices.append(tempDevInfo);
            emit pairedWithDevice(DeviceType::Wristwatch, tempDevInfo->getName());

        }else if(QString::compare(servInfo->getUuid(), crocsLeftServiceUUID) ==0){
            //CROCS_LEFT
            qDebug() << "Left croc connected" <<Qt::endl;
            for(int i = 0; i <pairedDevices.count(); ++i){
                qDebug() << "Index: "<< i <<Qt::endl;
                if(pairedDevices[i]->devType == DeviceType::CrocsLeft){
                    qDebug() << "This devType is already paired" << Qt::endl;

                    return;
                }
            }
            DeviceInfo *tempDevInfo = new DeviceInfo(initialScanTempDevice.getDevice());
            tempDevInfo->devType = DeviceType::CrocsLeft;
            pairedDevices.append(tempDevInfo);
            emit pairedWithDevice(DeviceType::CrocsLeft, tempDevInfo->getName());

        }else if(QString::compare(servInfo->getUuid(), crocsRightServiceUUID) ==0){
            //CROCS_RIGHT
            qDebug() << "Right croc connected" <<Qt::endl;
            for(int i = 0; i <pairedDevices.count(); ++i){
                qDebug() << "Index: "<< i <<Qt::endl;
                if(pairedDevices[i]->devType == DeviceType::CrocsRight){
                    qDebug() << "This devType is already paired" << Qt::endl;

                    return;
                }
            }
            DeviceInfo *tempDevInfo = new DeviceInfo(initialScanTempDevice.getDevice());
            tempDevInfo->devType = DeviceType::CrocsRight;
            pairedDevices.append(tempDevInfo);
            emit pairedWithDevice(DeviceType::CrocsRight, tempDevInfo->getName());

        }else if(QString::compare(servInfo->getUuid(),boxingGloveLeftServiceUUID) ==0){
            //BOXING_GLOVE_LEFT
            qDebug() << "Left boxing glove connected" <<Qt::endl;
            for(int i = 0; i <pairedDevices.count(); ++i){
                qDebug() << "Index: "<< i <<Qt::endl;
                if(pairedDevices[i]->devType == DeviceType::BoxingGloveLeft){
                    qDebug() << "This devType is already paired" << Qt::endl;

                    return;
                }
            }
            DeviceInfo *tempDevInfo = new DeviceInfo(initialScanTempDevice.getDevice());
            tempDevInfo->devType = DeviceType::BoxingGloveLeft;
            pairedDevices.append(tempDevInfo);
            emit pairedWithDevice(DeviceType::BoxingGloveLeft, tempDevInfo->getName());

        }else if(QString::compare(servInfo->getUuid(),boxingGloveRightServiceUUID) ==0){
            //BOXING_GLOVE_RIGHT
            qDebug() << "Right boxing glove connected" <<Qt::endl;
            for(int i = 0; i <pairedDevices.count(); ++i){
                qDebug() << "Index: "<< i <<Qt::endl;
                if(pairedDevices[i]->devType == DeviceType::BoxingGloveRight){
                    qDebug() << "This devType is already paired" << Qt::endl;

                    return;
                }
            }
            DeviceInfo *tempDevInfo = new DeviceInfo(initialScanTempDevice.getDevice());
            tempDevInfo->devType = DeviceType::BoxingGloveRight;
            pairedDevices.append(tempDevInfo);
            emit pairedWithDevice(DeviceType::BoxingGloveRight, tempDevInfo->getName());

        }

    }

    //Disconnect from device after pairing it up
    initialScanController->disconnectFromDevice();
}

bool BleController::isRandomAddress()
{
    return randomAddress;
}

void BleController::connectToPairedDevice(int devType){
    //Find this devType device in pairedDevices list
    DeviceInfo *currentDevice = nullptr;
    for(DeviceInfo *tempDevInfo : pairedDevices){
        if(tempDevInfo->devType == devType){
            currentDevice = tempDevInfo;
        }
    }
    if(currentDevice==nullptr){
        qDebug() << "Not found a device of this devType in paired device list..." <<Qt::endl;
        return;
    }
    QLowEnergyController *tempController = QLowEnergyController::createCentral(currentDevice->getDevice());
    controllers[devType] = tempController;

    startServiceScan(devType, currentDevice);
}

void BleController::startServiceScan(int devType, DeviceInfo *currentDevice){

    qDebug() << "Starting service scan for devType:" << devType <<Qt::endl; 
    qDebug() << "Doing service scan on device name: " <<currentDevice->getName() << Qt::endl;

    if (!currentDevice->getDevice().isValid()) {
        qWarning() << "Not a valid device";
        return;
    }

    //Clean up service and characterstic lists before assigning to them
    qDeleteAll(scannedCharacteristics[devType]);
    scannedCharacteristics[devType].clear();

    qDeleteAll(scannedServices[devType]);
    scannedServices[devType].clear();

    finalScannedServices[devType] = nullptr;

    qDebug() << "Done cleaning up" << Qt::endl;


    if(controllers[devType]!=nullptr){
        controllers[devType]->disconnectFromDevice();
        delete controllers[devType];
        controllers[devType] = nullptr;
    }


    // Connecting signals and slots for connecting to LE services.
    controllers[devType] = QLowEnergyController::createCentral(currentDevice->getDevice());
    connect(controllers[devType], &QLowEnergyController::connected,
            this, [=](){ BleController::serviceScanDeviceConnected(devType); });
        connect(controllers[devType], QOverload<QLowEnergyController::Error>::of(&QLowEnergyController::error), this, [=](QLowEnergyController::Error newErr){BleController::serviceScanErrorRecieved(newErr,devType);});
    connect(controllers[devType], &QLowEnergyController::disconnected,
            this, [=](){BleController::serviceScanDeviceDisconnected(devType);});
    connect(controllers[devType], &QLowEnergyController::serviceDiscovered, 
            this, [=](const QBluetoothUuid uuid){BleController::serviceScanAddLowEnergyService(uuid,devType);});
    connect(controllers[devType], &QLowEnergyController::discoveryFinished,
            this, [=](){BleController::serviceScanDone(devType);});

    if (isRandomAddress())
        controllers[devType]->setRemoteAddressType(QLowEnergyController::RandomAddress);

    else
        controllers[devType]->setRemoteAddressType(QLowEnergyController::PublicAddress);

    controllers[devType]->connectToDevice();
    currentDevice->connected = true;


    m_previousAddress = currentDevice->getAddress();

}


void BleController::serviceScanDeviceConnected(int devType)
{
    qDebug() << "Device connected" << devType << Qt::endl;
    connected = true;
    controllers[devType]->discoverServices();
}

void BleController::serviceScanErrorRecieved(QLowEnergyController::Error, int devType)
{
    qWarning() << "Error: " << controllers[devType]->errorString() << "DevType: " <<devType;
}

void BleController::serviceScanDeviceDisconnected(int devType)
{
    qWarning() << "Disconnect from device";
    for(DeviceInfo *devInfo: pairedDevices){
        if(devInfo->devType==devType){
            devInfo->connected = false;
        }
    }
    emit disconnectFromDevice(devType);
}

void BleController::serviceScanAddLowEnergyService(const QBluetoothUuid &serviceUuid, int devType)
{
    qDebug() << "New service found" << Qt::endl;
    QLowEnergyService *service = controllers[devType]->createServiceObject(serviceUuid);
    if (!service) {
        qWarning() << "Cannot create service for uuid";
        return;
    }
    auto serv = new ServiceInfo(service);
    scannedServices[devType].append(serv);

    emit servicesUpdated();
    ServiceInfo* tempInfo = qobject_cast<ServiceInfo *>(serv);
    qDebug() << "ServiceName: " << tempInfo->getName() << Qt::endl;
}

void BleController::serviceScanDone(int devType)
{
    qDebug() << "Service scan done... Seeing if device is compatible";
    // QString serviceUuid = scannedServices[devType][1]->getUuid();
    // qDebug() << "Main service UUID: " << serviceUuid << Qt::endl;

    for(ServiceInfo *servInfo: scannedServices[devType]){
        //Determine device type by the main service uuid
        if(QString::compare(servInfo->getUuid(), wristwatchServiceUUID) == 0){
            //WRISTWATCH
            qDebug() << "Scanning wristwatch device services" << Qt::endl;
            connectToMainService(servInfo,DeviceType::Wristwatch);
            return;

        }else if(QString::compare(servInfo->getUuid(), crocsLeftServiceUUID) ==0){
            //CROCS_LEFT
            qDebug() << "Scanning left crocs device services" << Qt::endl;
            connectToMainService(servInfo,DeviceType::CrocsLeft);
            return;

        }else if(QString::compare(servInfo->getUuid(), crocsRightServiceUUID) ==0){
            //CROCS_RIGHT
            qDebug() << "Scanning right crocs device services" << Qt::endl;
            connectToMainService(servInfo,DeviceType::CrocsRight);
            return;

        }else if(QString::compare(servInfo->getUuid(),boxingGloveLeftServiceUUID) ==0){
            //BOXING_GLOVE_LEFT
            qDebug() << "Scanning left boxing glove device services" << Qt::endl;
            connectToMainService(servInfo,DeviceType::BoxingGloveLeft);
            return;

        }else if(QString::compare(servInfo->getUuid(),boxingGloveRightServiceUUID) ==0){
            //BOXING_GLOVE_RIGHT
            qDebug() << "Scanning right boxing glove device services" << Qt::endl;
            connectToMainService(servInfo,DeviceType::BoxingGloveRight);
            return;

        }
    }
    //Unrecognizable device
    qDebug() << "THIS DEVICE CANT BE RECOGNIZED" << Qt::endl;
    return;

    //Disconnect from device after pairing it up
    // controllers[devType]->disconnectFromDevice();
}


void BleController::connectToMainService(ServiceInfo * serviceInformation, int devType)
{
    qDebug() << "Connecting to main service for devType: " <<devType <<Qt::endl;
    // ServiceInfo *serviceInformation = nullptr;
    // serviceInformation = scannedServices[devType][1];
    finalScannedServices[devType] = serviceInformation;
    QLowEnergyService *service = serviceInformation->service();

    //Only one custom service so connect to it...

    //WRISTWATCH
    //if(devType==0){
    //    //Make sure service has correct uuid
    //    if(serviceInformation->getUuid()==uuid){
    //        service = serviceInformation->service();
    //    }
    //}


    qDeleteAll(scannedCharacteristics[devType]);
    scannedCharacteristics[devType].clear();

    if (service->state() == QLowEnergyService::DiscoveryRequired) {
        connect(service, &QLowEnergyService::stateChanged,
                this, [=](QLowEnergyService::ServiceState newState){BleController::serviceDetailsDiscovered(newState,devType);});
        service->discoverDetails();
        return;
    }

    //discovery already done
    // const QList<QLowEnergyCharacteristic> chars = service->characteristics();
    // for (const QLowEnergyCharacteristic &ch : chars) {
    //     auto cInfo = new CharacteristicInfo(ch);
    //     scannedCharacteristics[devType].append(cInfo);
    // }
    setUpCharacteristicTracking(service,devType);
}

void BleController::serviceDetailsDiscovered(QLowEnergyService::ServiceState newState, int devType)
{
    qDebug() << "Service discovery state changed for devType: " << devType << Qt::endl;
    if (newState != QLowEnergyService::ServiceDiscovered) {
        if (newState == QLowEnergyService::DiscoveringServices) {
            qDebug() << "Still discovering services" << Qt::endl;
        }
        else if(newState== QLowEnergyService::InvalidService){
            qDebug() << "Invalid service..." <<Qt::endl;
        }else if(newState== QLowEnergyService::DiscoveryRequired){
            qDebug() << "Discovery required (something went wrong in serviceDetails discoery)..." <<Qt::endl;
    }
        return;
    }

    qDebug() << "Services successfully discoved" <<Qt::endl;

    QLowEnergyService *service = qobject_cast<QLowEnergyService *>(sender());
    if (!service)
        return;


    setUpCharacteristicTracking(service,devType);


}

void BleController::setUpCharacteristicTracking(QLowEnergyService *service, int devType){
    const QList<QLowEnergyCharacteristic> chars = service->characteristics();
    for (const QLowEnergyCharacteristic &ch : chars) {
        auto cInfo = new CharacteristicInfo(ch);
        qDebug() << ch.value() << Qt::endl;
        scannedCharacteristics[devType].append(cInfo);
        QLowEnergyDescriptor tempDescriptor = ch.descriptor(QBluetoothUuid::ClientCharacteristicConfiguration);
        if (tempDescriptor.isValid()) {
            qDebug() << "Writing to descriptor (enabling notifications)" << Qt::endl;
            service->writeDescriptor(tempDescriptor, QByteArray::fromHex("0100"));
        }
    }

    //Connect characteristic changed notification (MAKE SURE TO ENALBE THEM ON DEVICE)to a callback
    connect(service, &QLowEnergyService::characteristicChanged,
                this, [=](QLowEnergyCharacteristic newChar, QByteArray newVal){BleController::onCharacteristicChanged(newChar,newVal,devType);});

}

void BleController::onCharacteristicChanged(QLowEnergyCharacteristic characteristic, QByteArray newVal, int devType){
    qDebug() << "Characteristic changed for devType: " << devType << " and uuid: " << characteristic.uuid().toString() << Qt::endl;
    QString charUuid = characteristic.uuid().toString();
    charUuid.remove("{");
    charUuid.remove("}");
    if(devType==DeviceType::Wristwatch){
        if(QString::compare(charUuid,wristwatchGpsCharUUID)==0){
            qDebug() << "GPS char updated" << Qt::endl;
            double newGpsData;
            *(uint8_t*)&newGpsData = newVal.at(1);
            *(uint8_t*)(&(newGpsData)+8) = newVal.at(0);
            // for(unsigned char byte: newVal){
            //     qDebug() << byte << " ";
            // }
            
            // // float distanceInMeters
            // qDebug() << newFrame.test1 << newFrame.test2 << newFrame.test3 << newFrame.test4 <<Qt::endl;
            emit gpsDataRecieved(newGpsData);
            
        }
        else if(QString::compare(charUuid,wristwatchStepCharUUID)==0){
            uint32_t stepsNum  = 0; 
            // *( &stepsNum ) = (uint8_t)newVal.at(0);
            // *( &stepsNum+8 ) = (uint8_t)newVal.at(1);
            // *( &stepsNum+16 ) = (uint8_t)newVal.at(2);
            // *( &stepsNum+24 ) = (uint8_t)newVal.at(3);
            stepsNum  = ((uint8_t)newVal.at(3) << 24| (uint8_t)newVal.at(2) <<16 | (uint8_t)newVal.at(1) << 8 | (uint8_t)newVal.at(0));
            qDebug() << "DATA0:" <<(uint8_t)newVal.at(0) << Qt::endl;
            qDebug() << "DATA1:" <<(uint8_t)newVal.at(1) << Qt::endl;
            qDebug() << "DATA2:" <<(uint8_t)newVal.at(2) << Qt::endl;
            qDebug() << "DATA3:" << (uint8_t)newVal.at(3) << Qt::endl;
            qDebug() << "Step counter char updated, steps: " << stepsNum << Qt::endl;
            emit stepDataRecieved(stepsNum);
        }
        else if(QString::compare(charUuid,wristwatchHeartCharUUID)==0){
            uint8_t heartRateBPM = (uint8_t)newVal.at(0);
            qDebug() << "Heart rate char updated to" << heartRateBPM << Qt::endl;
            emit heartRateDataRecieved(heartRateBPM);
        }
    }else if(devType==DeviceType::CrocsLeft){
        if(QString::compare(charUuid,crocsAccelerometerCharUUID)==0){
            qDebug() << "Left croc accelerometer char updated" << Qt::endl;
            uint8_t accelerometerByte = newVal.at(0);
            bool isLeft = ((accelerometerByte)&(1<<7)) && ((accelerometerByte)&(1<<6));
            bool isRight = ((accelerometerByte)&(1<<5)) && ((accelerometerByte)&(1<<4));
            bool isBack = ((accelerometerByte)&(1<<3)) && ((accelerometerByte)&(1<<2));
            bool isFront = ((accelerometerByte)&(1<<1)) && ((accelerometerByte)&(1));

            qDebug() << "Accelerometer char(front: " <<isFront <<", back: " <<isBack<<", right: " <<isRight << ",left: " <<isLeft;

            emit leftCrocAccelerometerDataRecieved(isFront,isBack,isRight,isLeft);
        }
    }else if (devType==DeviceType::CrocsRight){
        if(QString::compare(charUuid,crocsAccelerometerCharUUID)==0){
            qDebug() << "Right croc accelerometer char updated" << Qt::endl;
            uint8_t accelerometerByte = newVal.at(0);
            bool isLeft = ((accelerometerByte)&(1<<7)) && ((accelerometerByte)&(1<<6));
            bool isRight = ((accelerometerByte)&(1<<5)) && ((accelerometerByte)&(1<<4));
            bool isBack = ((accelerometerByte)&(1<<3)) && ((accelerometerByte)&(1<<2));
            bool isFront = ((accelerometerByte)&(1<<1)) && ((accelerometerByte)&(1));

            qDebug() << "Accelerometer char(front: " <<isFront <<", back: " <<isBack<<", right: " <<isRight << ",left: " <<isLeft;

            emit rightCrocAccelerometerDataRecieved(isFront,isBack,isRight,isLeft);
        }
    }else if (devType==DeviceType::BoxingGloveLeft){
        if(QString::compare(charUuid,boxingGlovePunchCounterCharUUID)==0){
            qDebug() << "Left boxing glove punch counter char updated" << Qt::endl;
            uint8_t punchDirectionByte = newVal.at(0);
            bool isJab = false;
            bool isChop = false;
            bool isUppercut = false;
            if(punchDirectionByte==12){
                isJab = true;
            }else if(punchDirectionByte==48){
                isChop = true;
            }else if(punchDirectionByte==142){
                isUppercut = true;
            }
            qDebug() << "Left boxing glove punch: " << punchDirectionByte <<Qt::endl;
            emit leftPunchDetected(isJab,isChop,isUppercut);
        }
    }else if(devType==DeviceType::BoxingGloveRight){
        if(QString::compare(charUuid,boxingGlovePunchCounterCharUUID)==0){
            qDebug() << "Right boxing glove punch counter char updated" << Qt::endl;
            uint8_t punchDirectionByte = newVal.at(0);
            bool isJab = false;
            bool isChop = false;
            bool isUppercut = false;
            if(punchDirectionByte==12){
                isJab = true;
            }else if(punchDirectionByte==48){
                isChop = true;
            }else if(punchDirectionByte==142){
                isUppercut = true;
            }
            qDebug() << "Right boxing glove punch: " << punchDirectionByte <<Qt::endl;
            emit rightPunchDetected(isJab,isChop,isUppercut);
        }
    }
}

//TODO: No encryption here. We just save this device's name into a persitent(TODO) list
void BleController::setDeviceAsPaired(int index){
    DeviceInfo *device = qobject_cast<DeviceInfo*>(scannedDevices[index]);
    qDebug() << "Setting device: " << device->getName() << " as paired" << Qt::endl;
    //Check that sevice UUIDs match
}

bool BleController::getPairedDeviceConnectionStatus(int devType){
    for(DeviceInfo *devInfo: pairedDevices){
        if(devInfo->devType==devType){
            return devInfo->connected;
        }
    }
    return false;
}

QString BleController::getDeviceInfo(int devType){
    if(devType==DeviceType::Wristwatch){
        return wristwatchInfo;
    }else if(devType==DeviceType::CrocsLeft || devType==DeviceType::CrocsRight){
        return crocsInfo;
    }else if(devType==DeviceType::BoxingGloveLeft || devType==DeviceType::BoxingGloveRight){
        return boxingInfo;
    }
}

void BleController::unpairDevice(int devType){
    qDebug() << "Unpairing a device" << Qt::endl;
    int indexToRemove = -1;
    for(int i = 0; i <pairedDevices.count(); ++i){
        if(pairedDevices[i]->devType==devType){
            indexToRemove = i;
        }
    }

    if(indexToRemove!=-1){
        pairedDevices.removeAt(indexToRemove);
    }

    for(DeviceInfo *devInfo:pairedDevices){
        qDebug() << "Paired device: " << devInfo->devType << Qt::endl;
    }
}

void BleController::disconnectDevice(int devType){
    qDebug() << "Disconnecting from device devType: " << devType <<Qt::endl;
    controllers[devType]->disconnectFromDevice();
    for(DeviceInfo *devInfo: pairedDevices){
        if(devInfo->devType==devType){
            devInfo->connected = false;
        }
    }
    emit disconnectFromDevice(devType);
}

void BleController::startBoxingGame(){
    qDebug() << "Starting boxing game" << Qt::endl;
    toggleGameCharacteristic(DeviceType::BoxingGloveLeft, 2, 1);
    toggleGameCharacteristic(DeviceType::BoxingGloveRight, 2, 1);
}

void BleController::stopBoxingGame(){
    qDebug() << "Finishing boxing game" << Qt::endl;
    toggleGameCharacteristic(DeviceType::BoxingGloveLeft, 2, 0);
    toggleGameCharacteristic(DeviceType::BoxingGloveRight, 2, 0);
}

void BleController::startWalkingGame(){
    qDebug() << "Starting walking game" << Qt::endl;
    toggleGameCharacteristic(DeviceType::Wristwatch, 0, 1);
}

void BleController::stopWalkingGame(){
    qDebug() << "Stopping walking game" << Qt::endl;
    toggleGameCharacteristic(DeviceType::Wristwatch, 0, 0);
}

void BleController::startRunningGame(){
    qDebug() << "Starting running game" << Qt::endl;
    toggleGameCharacteristic(DeviceType::Wristwatch, 1, 1);

}

void BleController::stopRunningGame(){
    qDebug() << "Stopping running game" << Qt::endl;
    toggleGameCharacteristic(DeviceType::Wristwatch, 1, 0);

}

void BleController::startDancingGame(){
    qDebug() << "Starting dancing game" << Qt::endl;
    toggleGameCharacteristic(DeviceType::CrocsLeft, 3, 1);
    toggleGameCharacteristic(DeviceType::CrocsRight, 3, 1);
}

void BleController::stopDancingGame(){
    qDebug() << "Stopping dancing game" << Qt::endl;
    toggleGameCharacteristic(DeviceType::CrocsLeft,3,0);
    toggleGameCharacteristic(DeviceType::CrocsRight,3,0);

}
//gameType = 0 (walking game) 1(running game) 2(boxing game) 3(dancing game) 4(mindfulness game)
void BleController::toggleGameCharacteristic(int devType, int gameType, bool gameState){
    qDebug() << "Toggling game characteristic to: " <<gameState << Qt::endl;
    if(finalScannedServices[devType]==nullptr){
        qDebug() <<"Cant toggle game characterstic, corresponding device is not connected"<<Qt::endl;
        return;
    }

    //Find game toggle characteristic UUID for specific game type
    QString targetDeviceGameToggleCharUuid;
    if(gameType==0){
        targetDeviceGameToggleCharUuid = wristwatchWalkingGameToggleCharUUID;
    }else if(gameType==1){
        targetDeviceGameToggleCharUuid = wristwatchRunningGameToggleCharUUID;
    }else if(gameType==2){
        targetDeviceGameToggleCharUuid = boxingGloveBoxingGameToggleCharUUID;
    }else if(gameType==3){
        targetDeviceGameToggleCharUuid = crocsDancingGameToggleCharUUID;
    }else if(gameType==4){
        targetDeviceGameToggleCharUuid = wristwatchMindfulnessGameToggleCharUUID;
    }

    //Find if this characteristic exists on the device
    for(CharacteristicInfo *charInfo : scannedCharacteristics[devType]){
        qDebug() << charInfo->getUuid();
        if(QString::compare(charInfo->getUuid(),targetDeviceGameToggleCharUuid)==0){
            //Found game toggle characteristics
            QByteArray newData;
            newData.append((char)gameState);
            finalScannedServices[devType]->service()->writeCharacteristic(charInfo->getCharacteristic(),newData);
        }
    }


}

void BleController::emitLeftPunchDetected(bool isJab,bool isChop, bool isUppercut){
    emit leftPunchDetected(isJab,isChop,isUppercut);
}

void BleController::emitRightPunchDetected(bool isJab,bool isChop, bool isUppercut){
    emit rightPunchDetected(isJab,isChop,isUppercut);
}
void BleController::emitHeartRateDataRecieved(int heartRate){
    qDebug() << heartRate << Qt::endl;
    emit heartRateDataRecieved(heartRate);
}
void BleController::emitStepDataRecieved(int stepsNum){
    emit stepDataRecieved(stepsNum);
}
void BleController::emitGpsDataRecivied(double distanceInMeters){
    emit gpsDataRecieved(distanceInMeters);
}

bool BleController::checkWristwatchConnected(){
    for(DeviceInfo *devInfo:pairedDevices){
        if(devInfo->devType==DeviceType::Wristwatch && devInfo->connected){
            qDebug() << "Wristwatch is connected!" <<Qt::endl;
            return true;
        }else{
            qDebug() << "Wristwatch is not connected!" <<Qt::endl;
            return false;
        }
    }
    return false;
}
bool BleController::checkLeftCrocConnected(){
    for(DeviceInfo *devInfo:pairedDevices){
        if(devInfo->devType==DeviceType::CrocsLeft && devInfo->connected){
            qDebug() << "Left croc is connected!" <<Qt::endl;
            return true;
        }else{
            qDebug() << "Left croc is not connected!" <<Qt::endl;
            return false;
        }
    }
    return false;
}

bool BleController::checkRightCrocConnected(){
    for(DeviceInfo *devInfo:pairedDevices){
        if(devInfo->devType==DeviceType::CrocsRight && devInfo->connected){
            qDebug() << "Right croc is connected!" <<Qt::endl;
            return true;
        }else{
            qDebug() << "Right croc is not connected!" <<Qt::endl;
            return false;
        }
    }
    return false;
}
bool BleController::checkLeftBoxingGloveConnected(){
    for(DeviceInfo *devInfo:pairedDevices){
        if(devInfo->devType==DeviceType::BoxingGloveLeft && devInfo->connected){
            qDebug() << "Left boxing glove is connected!" <<Qt::endl;
            return true;
        }else{
            qDebug() << "Left boxing glove is not connected!" <<Qt::endl;
            return false;
        }
    }
    return false;
}
bool BleController::checkRightBoxingGloveConnected(){
    for(DeviceInfo *devInfo:pairedDevices){
        if(devInfo->devType==DeviceType::BoxingGloveRight && devInfo->connected){
            qDebug() << "Right boxing glove is connected!" <<Qt::endl;
            return true;
        }else{
            qDebug() << "Right boxing glove is not connected!" <<Qt::endl;
            return false;
        }
    }
    return false;
}
